<?php
/*--------------------------------------------------------------
   GambioRevenueCount.php 2021-08-16
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2021 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition;

use Doctrine\DBAL\Connection;
use Gambio\Admin\Modules\Statistics\App\Data\Factory as DataFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory as OverviewFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory\Option\Predefined\TimespanDropdown;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Collections\Options;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data;
use NumberFormatter;

class GambioRevenueCount extends GambioWidgetDefinition
{
    private const TYPE = 'gambio.revenue.count';
    
    private const VERSION = '0.1.0';
    
    private const ICON = "data:image/png;base64,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";
    
    private const WIDGET_TITLE = [
        self::LANGUAGE_CODE_GERMAN  => 'Umsatz',
        self::LANGUAGE_CODE_ENGLISH => 'Turnover'
    ];
    
    /**
     * @var DataFactory
     */
    private $dataFactory;
    
    /**
     * @var Connection
     */
    private $connection;
    
    /**
     * @var NumberFormatter
     */
    private $numberFormatter;
    
    
    /**
     * @inheritDoc
     */
    public function __construct(
        OverviewFactory $overviewFactory,
        DataFactory $dataFactory,
        Connection $connection,
        NumberFormatter $numberFormatter
    ) {
        $this->dataFactory     = $dataFactory;
        $this->connection      = $connection;
        $this->numberFormatter = $numberFormatter;
        
        parent::__construct($overviewFactory->createType(self::TYPE),
                            $overviewFactory->createVersion(self::VERSION),
                            $overviewFactory->createIconUsingData(self::ICON),
                            $overviewFactory->useVisualizations()->createText(),
                            $overviewFactory->useOptions()->createOptions($overviewFactory->useOptions()
                                                                              ->usePredefined()
                                                                              ->createTimespanDropdownIncludingToday($overviewFactory)),
                            $overviewFactory->createTitles($overviewFactory->createTitle($overviewFactory->createLanguageCode(self::LANGUAGE_CODE_GERMAN),
                                                                                         self::WIDGET_TITLE[self::LANGUAGE_CODE_GERMAN]),
                                                           $overviewFactory->createTitle($overviewFactory->createLanguageCode(self::LANGUAGE_CODE_ENGLISH),
                                                                                         self::WIDGET_TITLE[self::LANGUAGE_CODE_ENGLISH])));
    }
    
    
    /**
     * @inheritDoc
     */
    public function data(Options $options): Data
    {
        $timespan   = $this->dataFactory->useTimespan()->createFromTerm($options->getById(TimespanDropdown::ID)
                                                                            ->value());
        $parameters = [
            ':excludedOrderIds' => implode(',', self::EXCLUDED_ORDER_STATUS_IDS),
            ':startDate'        => $timespan->startDate()->format(self::QUERY_TIMESPAN_FORMAT_DATE_START),
            ':endDate'          => $timespan->endDate()->format(self::QUERY_TIMESPAN_FORMAT_DATE_END)
        ];
        
        $revenue    = $this->connection->createQueryBuilder()
                          ->select([
                                       'SUM(ROUND(orders_total.value / orders.currency_value, 2)) AS sales',
                                       'orders.currency AS currency'
                                   ])
                          ->from('orders')
                          ->join('orders',
                                 'orders_total',
                                 'orders_total',
                                 'orders.orders_id = orders_total.orders_id')
                          ->where('orders_total.class = "ot_total"')
                          ->andWhere('orders_status NOT IN (:excludedOrderIds)')
                          ->andWhere('orders.date_purchased BETWEEN :startDate AND :endDate')
                          ->setParameters($parameters)
                          ->groupBy('currency')
                          ->execute()
                          ->fetchAll()[0];
        
        $taxes = $this->connection->createQueryBuilder()
                     ->select([
                                  'SUM(ROUND(orders_total.value / orders.currency_value, 2)) AS taxes',
                                  'orders.currency AS currency'
                              ])
                     ->from('orders')
                     ->join('orders',
                            'orders_total',
                            'orders_total',
                            'orders.orders_id = orders_total.orders_id')
                     ->where('orders_total.class="ot_tax"')
                     ->andWhere('orders_status NOT IN (:excludedOrderIds)')
                     ->andWhere('orders.date_purchased BETWEEN :startDate AND :endDate')
                     ->setParameters($parameters)
                     ->groupBy('currency')
                     ->execute()
                     ->fetchAll()[0];
        
        $currency = $revenue['currency'] ?? 'EUR';
        $sales    = '0';
        
        if (isset($taxes['currency'], $revenue['sales']) && $taxes['currency'] === $currency) {
            $sales = (float)$revenue['sales'] - (float)$taxes['taxes'];
        } elseif (isset($revenue['sales'])) {
            $sales = (float)$revenue['sales'];
        }
        
        return $this->dataFactory->useTextData()->createTextData($this->dataFactory->useTextData()
                                                                     ->createValue($this->numberFormatter->formatCurrency((float)$sales,
                                                                                                                          $currency)));
    }
}
